/*
 * cBufferManager.cpp
 *
 * Manages buffers and free memory.
 */
#include "cBufferManager.h"
#include "cMsgBuffer.h"
#include "Util/gError.h"
#include "Util/cStack.h"
#include <malloc.h>
#include <stdio.h>


unsigned int cBufferManager::mNumAlloc    = 0;
unsigned int cBufferManager::mNumFree     = 0;
unsigned int cBufferManager::mBufferAlloc = 0;
unsigned int cBufferManager::mBufferFree  = 0;
cStack cBufferManager::mStack = cStack(DEFAULT_BUFFER_CACHE_SIZE);

void cBufferManager::PrintStats(ostream &s)
{
	s << endl;
	s << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;
	s << "Buffer Manager Stats    " << endl;
	s << "Num Alloced:  " << mNumAlloc << endl;
	s << "Num Freed:    " << mNumFree  << endl;
	s << "Buffer Alloc: " << mBufferAlloc << endl;
	s << "Buffer Free:  " << mBufferFree << endl;
	s << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;
}

/*
 * cBufferManager::GetBuffer()
 *
 * Purpose:	Gets a message buffer for use.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	A message buffer is pulled off the free list.
 * Return:	NULL if fail, else a valid pointer to a message buffer.
 */
cMsgBuffer*	cBufferManager::GetBuffer()
{
	cMsgBuffer *retVal = NULL;
	if( !(mStack.Pop((void **)&retVal)) )
	{
		retVal = new cMsgBuffer();	// Try to make a new one.
	}

	if(retVal)
	{
		mBufferAlloc++;
	}
	else
	{
		gError("Unable to get a new buffer!!!", __LINE__, __FILE__);
	}
	return retVal;
}

/*
 * cBufferManager::ReturnBuffer()
 *
 * Purpose:	Returns a buffer that is no longer in use (i.e. reference count went to 0)
 * IN:		msgBuffer	->	The buffer to return.
 * OUT:		-
 * Cond:	-
 * PostCnd:	A message buffer is added to the free list, or deleted if there is excess.
 * Return:	true if success, else false.
 */
bool cBufferManager::ReturnBuffer(cMsgBuffer* msgBuffer)
{
	bool retVal;
	mBufferFree++;
	retVal = mStack.Push((void *)msgBuffer);
	if(!retVal)
	{
		delete msgBuffer;
	}
	return retVal;
}

/*
 * cBufferManager::Alloc()
 *
 * Purpose:	Allocates a block of memory.
 * IN:		size	-> The size (in bytes) of memory to allocate.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The memory is allocated off the heap.
 * Return:	NULL if failure, else a valid pointer.
 */
void* cBufferManager::Alloc(unsigned int size)
{
	mNumAlloc++;
	return malloc(size);
}	

/*
 * cBufferManager::Free()
 *
 * Purpose:	Frees the given block of memory.
 * IN:		memblock	-> A pointer to the memory to be freed
 * OUT:		-
 * Cond:	-
 * PostCnd:	The memory is added to the heap.
 * Return:	-
 */
void cBufferManager::Free(void* memblock)
{
	mNumFree++;
	free(memblock);
}

int	cBufferManager::NumFreeBuffers()
{
	return mStack.GetNumElements();
}